from __future__ import annotations

from typing import Optional, Tuple, List
from django.db.models import Q
from accounts.models import User

def _csv_tokens(value: str) -> List[str]:
    return [t.strip().lower() for t in (value or "").split(",") if t.strip()]

def score_tutor_for_booking(tutor: User, subject: str, exam: str, service: str) -> float:
    """Simple heuristic scoring."""
    score = 0.0
    subj = (subject or "").strip().lower()
    ex = (exam or "").strip().lower()
    svc = (service or "").strip().lower()

    tutor_subjects = _csv_tokens(getattr(tutor, "subjects", ""))
    tutor_exams = _csv_tokens(getattr(tutor, "exams", ""))
    tutor_services = _csv_tokens(getattr(tutor, "services", ""))

    if subj:
        for t in tutor_subjects:
            if t == subj:
                score += 3.0
            elif t and t in subj:
                score += 1.5
            elif subj in t:
                score += 1.0

    if ex:
        for t in tutor_exams:
            if t == ex:
                score += 2.0
            elif t and t in ex:
                score += 1.0
            elif ex in t:
                score += 0.8

    if svc:
        for t in tutor_services:
            if t == svc:
                score += 1.0
            elif t and (t in svc or svc in t):
                score += 0.6

    return score

def find_best_tutor(subject: str, exam: str, service: str) -> Optional[User]:
    """Return best matching tutor or None."""
    # Broad prefilter to reduce scanning (contains subject/exam strings)
    qs = User.objects.filter(role=User.ROLE_TUTOR, is_active=True)
    subj = (subject or "").strip()
    ex = (exam or "").strip()
    svc = (service or "").strip()
    if subj or ex or svc:
        q = Q()
        if subj:
            q |= Q(subjects__icontains=subj)
        if ex:
            q |= Q(exams__icontains=ex)
        if svc:
            q |= Q(services__icontains=svc)
        qs = qs.filter(q) if q else qs

    best = None
    best_score = 0.0
    for tutor in qs[:200]:
        s = score_tutor_for_booking(tutor, subject, exam, service)
        if s > best_score:
            best_score = s
            best = tutor

    return best if best_score > 0 else None
