from rest_framework import viewsets, permissions
from rest_framework.decorators import action
from rest_framework.response import Response
from django.shortcuts import get_object_or_404
from .models import Conversation, Message
from .serializers import ConversationSerializer, MessageSerializer
from bookings.models import Booking

class IsConversationMember(permissions.BasePermission):
    def has_object_permission(self, request, view, obj: Conversation):
        if request.user.is_staff:
            return True
        return obj.student_id == request.user.id or obj.tutor_id == request.user.id

class ConversationViewSet(viewsets.ReadOnlyModelViewSet):
    permission_classes = [permissions.IsAuthenticated, IsConversationMember]
    serializer_class = ConversationSerializer

    def get_queryset(self):
        u = self.request.user
        if u.is_staff:
            return Conversation.objects.all().order_by("-created_at")
        return Conversation.objects.filter(student=u) | Conversation.objects.filter(tutor=u)

    @action(detail=True, methods=["get"])
    def messages(self, request, pk=None):
        conv = self.get_object()
        qs = conv.messages.all()
        return Response(MessageSerializer(qs, many=True).data)

    @action(detail=True, methods=["post"])
    def send(self, request, pk=None):
        conv = self.get_object()
        text = (request.data.get("text") or "").strip()
        if not text:
            return Response({"detail": "Message text is required."}, status=400)
        msg = Message.objects.create(conversation=conv, sender=request.user, text=text)
        return Response(MessageSerializer(msg).data, status=201)

    @action(detail=False, methods=["post"])
    def start_for_booking(self, request):
        """Ensure a conversation exists for a booking (student/tutor only)."""
        booking_id = request.data.get("booking_id")
        booking = get_object_or_404(Booking, id=booking_id)
        if not (request.user.is_staff or booking.student_id == request.user.id or booking.tutor_id == request.user.id):
            return Response({"detail": "Not allowed."}, status=403)
        if not booking.tutor_id:
            return Response({"detail": "Booking has no tutor assigned yet."}, status=400)
        conv, _ = Conversation.objects.get_or_create(
            booking=booking,
            defaults={"student": booking.student, "tutor": booking.tutor},
        )
        return Response(ConversationSerializer(conv).data)
