from django.conf import settings
from django.db import models
from bookings.models import Booking

class Conversation(models.Model):
    booking = models.OneToOneField(Booking, on_delete=models.CASCADE, related_name="conversation")
    student = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE, related_name="conversations_as_student")
    tutor = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE, related_name="conversations_as_tutor")
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"Conversation for Booking #{self.booking_id}"

class Message(models.Model):
    conversation = models.ForeignKey(Conversation, on_delete=models.CASCADE, related_name="messages")
    sender = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE, related_name="messages_sent")
    text = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    is_read = models.BooleanField(default=False)

    class Meta:
        ordering = ["created_at"]

    def __str__(self):
        return f"Msg #{self.id} in Conv #{self.conversation_id}"


class SupportThread(models.Model):
    """A support inbox thread for ET-chat.
    - If user is authenticated, we tie it to user.
    - Otherwise we tie it to a session_key (guest).
    """
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True, related_name="support_threads")
    session_key = models.CharField(max_length=64, blank=True, default="", db_index=True)
    assigned_staff = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True, related_name="assigned_support_threads")
    status = models.CharField(max_length=20, default="open", db_index=True)  # open / closed
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        who = self.user_id or (self.session_key[:10] + "…")
        return f"SupportThread #{self.id} ({who})"

class SupportMessage(models.Model):
    thread = models.ForeignKey(SupportThread, on_delete=models.CASCADE, related_name="messages")
    sender = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.SET_NULL, null=True, blank=True, related_name="support_messages_sent")
    sender_label = models.CharField(max_length=32, default="guest")  # guest / user / staff / system
    text = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    is_read = models.BooleanField(default=False)

    class Meta:
        ordering = ["created_at"]

    def __str__(self):
        return f"SupportMsg #{self.id} in Thread #{self.thread_id}"
