# -*- coding: utf-8 -*-

from pathlib import Path
import os

from dotenv import load_dotenv

# Optional: allow PyMySQL to act as MySQLdb (helps on shared hosting where mysqlclient fails)
try:
    import pymysql  # type: ignore
    pymysql.install_as_MySQLdb()
except Exception:
    pass

BASE_DIR = Path(__file__).resolve().parent.parent

# -----------------------------------------------------------------------------
# ENV loading (switchable for local dev vs production)
#
# Namecheap does NOT auto-detect env files. We support:
#   - .env              (shared defaults, optional)
#   - .env.local        (local development overrides)
#   - .env.production   (production overrides)
#
# Selection order:
#   1) If ENV_FILE is set -> load that only
#   2) Else load .env if present, then:
#        - if DJANGO_ENV in (prod/production/live) -> load .env.production (if present)
#        - otherwise -> load .env.local (if present)
#
# Local quick start:
#   copy .env.example to .env.local and keep DB_ENGINE=sqlite
# -----------------------------------------------------------------------------
ENV_FILE = os.getenv("ENV_FILE", "").strip()
DJANGO_ENV = os.getenv("DJANGO_ENV", "").strip().lower() or "local"

def _load_if_exists(path: Path, override: bool = True) -> bool:
    if path and path.exists():
        load_dotenv(path, override=override)
        return True
    return False

ACTIVE_ENV_FILES = []

if ENV_FILE:
    if _load_if_exists(BASE_DIR / ENV_FILE, override=True):
        ACTIVE_ENV_FILES.append(ENV_FILE)
else:
    if _load_if_exists(BASE_DIR / ".env", override=False):
        ACTIVE_ENV_FILES.append(".env")

    if DJANGO_ENV in ("prod", "production", "live"):
        if _load_if_exists(BASE_DIR / ".env.production", override=True):
            ACTIVE_ENV_FILES.append(".env.production")
    else:
        if _load_if_exists(BASE_DIR / ".env.local", override=True):
            ACTIVE_ENV_FILES.append(".env.local")


# --- Core env (must be defined before any DEBUG-dependent logic) ---
SECRET_KEY = os.getenv("SECRET_KEY", "dev-secret-key-change-me")
DEBUG = (os.getenv("DEBUG", "0").strip().lower() in ("1","true","yes","on"))

# Optional startup log (safe on shared hosting). Set ENV_STARTUP_LOG=1 to force.
if os.getenv("ENV_STARTUP_LOG", "0").strip().lower() in ("1","true","yes","on") or DEBUG:
    print(f"[ENV] Loaded {', '.join(ACTIVE_ENV_FILES) if ACTIVE_ENV_FILES else '(none)'} | DJANGO_ENV={DJANGO_ENV} | ALLOWED_HOSTS={os.getenv('ALLOWED_HOSTS','')} | DB_ENGINE={os.getenv('DB_ENGINE','')}")




def _env_list(name: str, default: str = ""):
    raw = os.getenv(name, default) or ""
    return [h.strip() for h in raw.split(",") if h.strip()]

def _db_engine(raw: str) -> str:
    raw = (raw or "").strip().lower()
    if not raw:
        return "django.db.backends.sqlite3"
    # Allow shorthand
    if raw in ("sqlite", "sqlite3"):
        return "django.db.backends.sqlite3"
    if raw in ("mysql", "mariadb"):
        return "django.db.backends.mysql"
    if raw in ("postgres", "postgresql", "psql"):
        return "django.db.backends.postgresql"
    # Already a full Django backend path?
    return os.getenv("DB_ENGINE", raw)


ALLOWED_HOSTS = _env_list("ALLOWED_HOSTS", "*") or ["*"]
CSRF_TRUSTED_ORIGINS = _env_list("CSRF_TRUSTED_ORIGINS", "")

INSTALLED_APPS = [
    "channels",
    "django.contrib.admin",
    "django.contrib.auth",
    "django.contrib.contenttypes",
    "django.contrib.sessions",
    "django.contrib.messages",
    "django.contrib.staticfiles",

    # third-party
    "rest_framework",

    # local
    "accounts",
    "bookings",
    "portal",
    "pages",
    "chat",
]

MIDDLEWARE = [
    "django.middleware.security.SecurityMiddleware",
    "whitenoise.middleware.WhiteNoiseMiddleware",
    "django.contrib.sessions.middleware.SessionMiddleware",
    "django.middleware.common.CommonMiddleware",
    "django.middleware.csrf.CsrfViewMiddleware",
    "django.contrib.auth.middleware.AuthenticationMiddleware",
    "django.contrib.messages.middleware.MessageMiddleware",
    "django.middleware.clickjacking.XFrameOptionsMiddleware",
]

ROOT_URLCONF = "enabled_tutors.urls"

TEMPLATES = [
    {
        "BACKEND": "django.template.backends.django.DjangoTemplates",
        "DIRS": [BASE_DIR / "templates"],
        "APP_DIRS": True,
        "OPTIONS": {
            "context_processors": [
                "django.template.context_processors.debug",
                "django.template.context_processors.request",
                "django.contrib.auth.context_processors.auth",
                "django.contrib.messages.context_processors.messages",
            ],
        },
    },
]

WSGI_APPLICATION = "enabled_tutors.wsgi.application"
ASGI_APPLICATION = 'enabled_tutors.asgi.application'

DATABASES = {
    "default": {
        "ENGINE": _db_engine(os.getenv("DB_ENGINE", "")),
        "NAME": os.getenv("DB_NAME", str(BASE_DIR / "db.sqlite3")),
        "USER": os.getenv("DB_USER", ""),
        "PASSWORD": os.getenv("DB_PASSWORD", ""),
        "HOST": os.getenv("DB_HOST", ""),
        "PORT": os.getenv("DB_PORT", ""),
    }
}
# -----------------------------------------------------------------------------
# MySQL hardening (utf8mb4) + simple connection pooling
# Uses Django persistent connections via CONN_MAX_AGE.
# Configure with:
#   DB_CONN_MAX_AGE=60   (seconds, default: 60 in production, 0 in local)
# -----------------------------------------------------------------------------
if DATABASES.get('default', {}).get('ENGINE') == 'django.db.backends.mysql':
    # Ensure utf8mb4 across the board
    DATABASES['default'].setdefault('OPTIONS', {})
    DATABASES['default']['OPTIONS'].setdefault('charset', 'utf8mb4')
    # Recommended strict mode (safe defaults; MySQL may ignore if not permitted)
    DATABASES['default']['OPTIONS'].setdefault('init_command', "SET sql_mode='STRICT_TRANS_TABLES'")

    # Simple pooling via persistent connections
    _env = os.getenv('DJANGO_ENV', 'local').strip().lower()
    _default_age = 60 if _env in ('prod','production','live') else 0
    try:
        DATABASES['default']['CONN_MAX_AGE'] = int(os.getenv('DB_CONN_MAX_AGE', str(_default_age)))
    except Exception:
        DATABASES['default']['CONN_MAX_AGE'] = _default_age



# -----------------------------------------------------------------------------
# DB automatic fallback (dev-safe): if MySQL is selected but credentials/host fail,
# fall back to SQLite so local dev can still boot.
# Control with:
#   DB_FALLBACK=1/0  (default: 1 for non-production, 0 for production)
#   DB_FALLBACK_SQLITE_PATH=db.sqlite3
# -----------------------------------------------------------------------------
def _truthy(v: str) -> bool:
    return str(v).strip().lower() in ("1", "true", "yes", "on")

def _should_fallback_to_sqlite() -> bool:
    env = os.getenv("DJANGO_ENV", "local").strip().lower()
    default = "0" if env == "production" else "1"
    return _truthy(os.getenv("DB_FALLBACK", default))

def _try_mysql_connection() -> bool:
    host = os.getenv("DB_HOST", "localhost").strip() or "localhost"
    try:
        port = int(os.getenv("DB_PORT", "3306"))
    except Exception:
        port = 3306
    user = os.getenv("DB_USER", "")
    pwd = os.getenv("DB_PASSWORD", "")
    name = os.getenv("DB_NAME", "")

    # Try mysqlclient first
    try:
        import MySQLdb  # type: ignore
        MySQLdb.connect(
            host=host, port=port, user=user, passwd=pwd, db=name,
            connect_timeout=3
        ).close()
        return True
    except Exception:
        pass

    # Try PyMySQL fallback (if installed)
    try:
        import pymysql  # type: ignore
        pymysql.connect(
            host=host, port=port, user=user, password=pwd, database=name,
            connect_timeout=3
        ).close()
        return True
    except Exception:
        return False

if DATABASES.get("default", {}).get("ENGINE") == "django.db.backends.mysql" and _should_fallback_to_sqlite():
    if not _try_mysql_connection():
        _sqlite_path = os.getenv("DB_FALLBACK_SQLITE_PATH", str(BASE_DIR / "db.sqlite3"))
        DATABASES["default"] = {
            "ENGINE": "django.db.backends.sqlite3",
            "NAME": _sqlite_path,
        }
        # Keep startup visible but non-fatal (ASCII-only for maximum hosting compatibility)
        print(f"WARNING: MySQL connection failed; falling back to SQLite at {_sqlite_path}")


AUTH_PASSWORD_VALIDATORS = [
    {"NAME": "django.contrib.auth.password_validation.UserAttributeSimilarityValidator"},
    {"NAME": "django.contrib.auth.password_validation.MinimumLengthValidator"},
    {"NAME": "django.contrib.auth.password_validation.CommonPasswordValidator"},
    {"NAME": "django.contrib.auth.password_validation.NumericPasswordValidator"},
]

LANGUAGE_CODE = "en-us"
TIME_ZONE = "UTC"
USE_I18N = True
USE_TZ = True

STATIC_URL = "/static/"
STATICFILES_DIRS = [BASE_DIR / "static"]
STATIC_ROOT = BASE_DIR / "staticfiles"

MEDIA_URL = "/media/"
MEDIA_ROOT = BASE_DIR / "media"

DEFAULT_AUTO_FIELD = "django.db.models.BigAutoField"

AUTH_USER_MODEL = "accounts.User"

LOGIN_URL = "accounts:login"
LOGIN_REDIRECT_URL = "portal:router"
LOGOUT_REDIRECT_URL = "pages:home"

AUTHENTICATION_BACKENDS = [
    "accounts.auth_backends.EmailOrUsernameBackend",
    "django.contrib.auth.backends.ModelBackend",
]

REST_FRAMEWORK = {
    "DEFAULT_AUTHENTICATION_CLASSES": [
        "rest_framework.authentication.SessionAuthentication",
        "rest_framework.authentication.BasicAuthentication",
    ],
    "DEFAULT_PERMISSION_CLASSES": [
        "rest_framework.permissions.IsAuthenticated",
    ],
}


if not DEBUG:
    STATICFILES_STORAGE = "whitenoise.storage.CompressedManifestStaticFilesStorage"

# -----------------------------------------------------------------------------
# Django Channels
# -----------------------------------------------------------------------------
# If REDIS_URL is set (e.g. redis://127.0.0.1:6379/0), we'll use Redis.
# Otherwise, we fall back to in-memory layer (works for single-process dev).
REDIS_URL = os.getenv("REDIS_URL", "").strip()

if REDIS_URL:
    CHANNEL_LAYERS = {
        "default": {
            "BACKEND": "channels_redis.core.RedisChannelLayer",
            "CONFIG": {"hosts": [REDIS_URL]},
        }
    }
else:
    CHANNEL_LAYERS = {
        "default": {"BACKEND": "channels.layers.InMemoryChannelLayer"}
    }


# -----------------------------------------------------------------------------
# Email (Notifications)
# -----------------------------------------------------------------------------
# In DEBUG we default to console backend to avoid SMTP errors locally.
EMAIL_BACKEND = os.getenv(
    "EMAIL_BACKEND",
    "django.core.mail.backends.console.EmailBackend" if DEBUG else "django.core.mail.backends.smtp.EmailBackend",
)
EMAIL_HOST = os.getenv("EMAIL_HOST", "")
EMAIL_PORT = int(os.getenv("EMAIL_PORT", "587"))
EMAIL_USE_TLS = os.getenv("EMAIL_USE_TLS", "1") in ("1", "true", "True", "yes", "on")
EMAIL_HOST_USER = os.getenv("EMAIL_HOST_USER", "")
EMAIL_HOST_PASSWORD = os.getenv("EMAIL_HOST_PASSWORD", "")
DEFAULT_FROM_EMAIL = os.getenv("DEFAULT_FROM_EMAIL", EMAIL_HOST_USER or "no-reply@example.com")
CONSULTATION_NOTIFY_EMAIL = os.getenv("CONSULTATION_NOTIFY_EMAIL", EMAIL_HOST_USER or "")

# Optional fallback staff email (used if a support thread isn't assigned yet)
SUPPORT_NOTIFY_STAFF_EMAIL = os.getenv("SUPPORT_NOTIFY_STAFF_EMAIL", "")


# Production security toggles (safe defaults)
SESSION_COOKIE_SECURE = os.getenv("SESSION_COOKIE_SECURE", "0").strip().lower() in ("1","true","yes","on")
CSRF_COOKIE_SECURE = os.getenv("CSRF_COOKIE_SECURE", "0").strip().lower() in ("1","true","yes","on")
SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")
