from django.shortcuts import render

def home(request):
    return render(request, "index.html")

def services(request):
    return render(request, "pages/services.html")

def exams(request):
    return render(request, "pages/exams.html")

def pricing(request):
    return render(request, "pages/pricing.html")

def about(request):
    return render(request, "pages/about.html")

def contact(request):
    return render(request, "pages/contact.html")

def exam_detail(request, slug):
    # map slug -> template file
    mapping = {
        "waec": "pages/exam-waec.html",
        "sat": "pages/exam-sat.html",
        "gcse": "pages/exam-gcse.html",
        "11plus": "pages/exam-11plus.html",
    }
    tmpl = mapping.get(slug)
    if not tmpl:
        return render(request, "pages/exams.html")
    return render(request, tmpl)


from django.http import Http404

def component_partial(request, name):
    allowed = {"header", "footer", "page-header", "bottom-nav"}
    if name not in allowed:
        raise Http404()
    return render(request, f"components/{name}.html")


from django.views.decorators.http import require_POST
from django.views.decorators.csrf import csrf_exempt
from django.http import JsonResponse
from django.conf import settings
from django.core.mail import EmailMessage
from .models import ConsultationRequest

def _get_post_value(request, key, default=""):
    # Accept JSON and form-encoded
    if request.content_type and "application/json" in request.content_type:
        try:
            import json
            data = json.loads(request.body.decode("utf-8") or "{}")
            return (data.get(key) or default).strip()
        except Exception:
            return default
    return (request.POST.get(key) or default).strip()

def _send_notify_email(subject, body, reply_to_email=""):
    notify_to = getattr(settings, "CONSULTATION_NOTIFY_EMAIL", "") or settings.EMAIL_HOST_USER
    if not notify_to:
        return False, "CONSULTATION_NOTIFY_EMAIL/EMAIL_HOST_USER not set"
    msg = EmailMessage(
        subject=subject[:180],
        body=body,
        from_email=getattr(settings, "DEFAULT_FROM_EMAIL", None),
        to=[notify_to],
        reply_to=[reply_to_email] if reply_to_email else [],
    )
    msg.send(fail_silently=False)
    return True, "sent"

@csrf_exempt
@require_POST
def consultation_submit(request):
    name = _get_post_value(request, "name")
    email = _get_post_value(request, "email")
    phone = _get_post_value(request, "phone")
    country = _get_post_value(request, "country")
    subject = _get_post_value(request, "subject", "Consultation Request")
    message = _get_post_value(request, "message")
    page_url = _get_post_value(request, "page_url") or request.META.get("HTTP_REFERER", "")

    if not name or not email:
        return JsonResponse({"ok": False, "error": "Name and email are required"}, status=400)

    obj = ConsultationRequest.objects.create(
        source="consultation",
        name=name,
        email=email,
        phone=phone,
        country=country,
        subject=subject,
        message=message,
        page_url=page_url,
    )

    body = f"""New Consultation Request

Name: {name}
Email: {email}
Phone: {phone}
Country: {country}
Subject: {subject}
Page: {page_url}

Message:
{message}

---
Request ID: {obj.id}
"""

    try:
        _send_notify_email(f"ENABLED Tutors • Consultation: {name}", body, reply_to_email=email)
    except Exception as e:
        return JsonResponse({"ok": False, "saved": True, "error": str(e)}, status=500)

    return JsonResponse({"ok": True, "saved": True})

@csrf_exempt
@require_POST
def lead_submit(request):
    name = _get_post_value(request, "name")
    email = _get_post_value(request, "email")
    phone = _get_post_value(request, "phone")
    subject = _get_post_value(request, "subject", "Lead")
    message = _get_post_value(request, "message")
    page_url = _get_post_value(request, "page_url") or request.META.get("HTTP_REFERER", "")

    if not name or not email:
        return JsonResponse({"ok": False, "error": "Name and email are required"}, status=400)

    obj = ConsultationRequest.objects.create(
        source="lead",
        name=name,
        email=email,
        phone=phone,
        country="",
        subject=subject,
        message=message,
        page_url=page_url,
    )

    body = f"""New Lead

Name: {name}
Email: {email}
Phone: {phone}
Subject: {subject}
Page: {page_url}

Message:
{message}

---
Lead ID: {obj.id}
"""

    try:
        _send_notify_email(f"ENABLED Tutors • New Lead: {name}", body, reply_to_email=email)
    except Exception as e:
        return JsonResponse({"ok": False, "saved": True, "error": str(e)}, status=500)

    return JsonResponse({"ok": True, "saved": True})
