package com.enabledtutors.app

import android.annotation.SuppressLint
import android.media.RingtoneManager
import android.net.Uri
import android.os.Bundle
import android.os.VibrationEffect
import android.os.Vibrator
import android.webkit.JavascriptInterface
import android.webkit.ValueCallback
import android.webkit.WebChromeClient
import android.webkit.WebResourceRequest
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.appcompat.app.AppCompatActivity
import com.google.android.material.floatingactionbutton.FloatingActionButton

class MainActivity : AppCompatActivity() {
    private lateinit var webView: WebView
    private var fileCallback: ValueCallback<Array<Uri>>? = null

    @SuppressLint("SetJavaScriptEnabled")
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)

        webView = findViewById(R.id.webview)

        val settings = webView.settings
        settings.javaScriptEnabled = true
        settings.domStorageEnabled = true
        settings.mediaPlaybackRequiresUserGesture = false
        settings.allowFileAccess = true
        settings.allowContentAccess = true
        settings.useWideViewPort = true
        settings.loadWithOverviewMode = true

        // JS bridge for sound + vibration when staff replies
        webView.addJavascriptInterface(AndroidBridge(), "AndroidBridge")

        webView.webViewClient = object : WebViewClient() {
            override fun shouldOverrideUrlLoading(view: WebView?, request: WebResourceRequest?): Boolean {
                // Keep everything inside the WebView
                return false
            }
        }

        webView.webChromeClient = object : WebChromeClient() {
            override fun onShowFileChooser(
                webView: WebView?,
                filePathCallback: ValueCallback<Array<Uri>>?,
                fileChooserParams: FileChooserParams?
            ): Boolean {
                fileCallback = filePathCallback
                val intent = fileChooserParams?.createIntent()
                return try {
                    startActivityForResult(intent, 1001)
                    true
                } catch (e: Exception) {
                    fileCallback = null
                    false
                }
            }
        }

        // Native ET-chat floating button (opens the web ET-chat widget)
        try {
            val root = findViewById<android.view.ViewGroup>(android.R.id.content)
            val etFab = FloatingActionButton(this).apply {
                contentDescription = "ET-chat"
                size = FloatingActionButton.SIZE_MINI
                setImageResource(android.R.drawable.ic_dialog_email)
                setOnClickListener { openEtChatNative() }
            }
            val lp = android.widget.FrameLayout.LayoutParams(
                android.widget.FrameLayout.LayoutParams.WRAP_CONTENT,
                android.widget.FrameLayout.LayoutParams.WRAP_CONTENT
            )
            lp.marginEnd = (18 * resources.displayMetrics.density).toInt()
            lp.bottomMargin = (120 * resources.displayMetrics.density).toInt()
            lp.gravity = android.view.Gravity.END or android.view.Gravity.BOTTOM
            root.addView(etFab, lp)
        } catch (_: Exception) {}

        val baseUrl = getString(R.string.base_url)
        webView.loadUrl(baseUrl)
    }

    @Deprecated("Deprecated in Java")
    override fun onActivityResult(requestCode: Int, resultCode: Int, data: android.content.Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if (requestCode == 1001) {
            val results = WebChromeClient.FileChooserParams.parseResult(resultCode, data)
            fileCallback?.onReceiveValue(results)
            fileCallback = null
        }
    }

    override fun onBackPressed() {
        if (webView.canGoBack()) webView.goBack() else super.onBackPressed()
    }

    private fun openEtChatNative() {
        try {
            webView.post {
                webView.evaluateJavascript(
                    """(function(){
                        try{
                          var fab = document.getElementById('supportFab');
                          if (fab) fab.click();
                          var open = document.getElementById('openEtChat');
                          if (open) open.click();
                        }catch(e){}
                      })();""",
                    null
                )
            }
        } catch (_: Exception) {}
    }

    private inner class AndroidBridge {
        @JavascriptInterface
        fun notifySupportReply() {
            runOnUiThread {
                try {
                    val uri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION)
                    val r = RingtoneManager.getRingtone(this@MainActivity, uri)
                    r?.play()
                } catch (_: Exception) {}

                try {
                    val vib = getSystemService(VIBRATOR_SERVICE) as Vibrator
                    if (android.os.Build.VERSION.SDK_INT >= 26) {
                        vib.vibrate(VibrationEffect.createOneShot(140, VibrationEffect.DEFAULT_AMPLITUDE))
                    } else {
                        @Suppress("DEPRECATION")
                        vib.vibrate(140)
                    }
                } catch (_: Exception) {}
            }
        }
    }
}
