from rest_framework import viewsets, permissions
from rest_framework.response import Response
from rest_framework.decorators import action
from django.utils import timezone
from .matching import find_best_tutor
from .models import Booking
from .serializers import BookingSerializer, BookingCreateSerializer

class IsOwnerOrTutor(permissions.BasePermission):
    def has_object_permission(self, request, view, obj: Booking):
        if request.user.is_staff:
            return True
        return obj.student_id == request.user.id or obj.tutor_id == request.user.id

class BookingViewSet(viewsets.ModelViewSet):
    permission_classes = [permissions.IsAuthenticated, IsOwnerOrTutor]

    def get_queryset(self):
        user = self.request.user
        if user.is_staff:
            return Booking.objects.all().order_by("-created_at")
        # Students: see their own; Tutors: see assigned
        if getattr(user, "role", None) == "tutor":
            return Booking.objects.filter(tutor=user).order_by("-created_at")
        return Booking.objects.filter(student=user).order_by("-created_at")

    def get_serializer_class(self):
        if self.action in ("create",):
            return BookingCreateSerializer
        return BookingSerializer

    def perform_create(self, serializer):
        booking = serializer.save(student=self.request.user)
        # Auto-match tutor (best-effort). Admin can reassign later.
        if booking.tutor_id is None:
            best = find_best_tutor(booking.subject, booking.exam, booking.service)
            if best:
                booking.tutor = best
                booking.status = Booking.STATUS_ASSIGNED
                booking.save(update_fields=["tutor","status","updated_at"])
                try:
                    from chat.models import Conversation
                    Conversation.objects.get_or_create(booking=booking, defaults={"student": booking.student, "tutor": booking.tutor})
                except Exception:
                    pass


    @action(detail=True, methods=["post"]) 
    def accept(self, request, pk=None):
        booking = self.get_object()
        user = request.user
        if getattr(user, "role", None) != "tutor":
            return Response({"detail": "Only tutors can accept."}, status=403)
        if booking.tutor_id != user.id:
            return Response({"detail": "Not assigned to you."}, status=403)
        from chat.models import Conversation
        Conversation.objects.get_or_create(booking=booking, defaults={"student": booking.student, "tutor": booking.tutor})
        booking.status = Booking.STATUS_TUTOR_ACCEPTED
        booking.tutor_responded_at = timezone.now()
        booking.tutor_response_note = (request.data.get("note") or "").strip()[:255]
        booking.save(update_fields=["status","tutor_responded_at","tutor_response_note","updated_at"])
        return Response(BookingSerializer(booking).data)

    @action(detail=True, methods=["post"]) 
    def decline(self, request, pk=None):
        booking = self.get_object()
        user = request.user
        if getattr(user, "role", None) != "tutor":
            return Response({"detail": "Only tutors can decline."}, status=403)
        if booking.tutor_id != user.id:
            return Response({"detail": "Not assigned to you."}, status=403)
        from chat.models import Conversation
        Conversation.objects.get_or_create(booking=booking, defaults={"student": booking.student, "tutor": booking.tutor})
        booking.status = Booking.STATUS_TUTOR_DECLINED
        booking.tutor_responded_at = timezone.now()
        booking.tutor_response_note = (request.data.get("note") or "").strip()[:255]
        booking.save(update_fields=["status","tutor_responded_at","tutor_response_note","updated_at"])
        return Response(BookingSerializer(booking).data)

    @action(detail=False, methods=["get"])
    def my(self, request):
        """Return my bookings (student) and assigned bookings (tutor)"""
        user = request.user
        student_qs = Booking.objects.filter(student=user).order_by("-created_at")
        tutor_qs = Booking.objects.filter(tutor=user).order_by("-created_at")
        return Response({
            "student": BookingSerializer(student_qs, many=True).data,
            "tutor": BookingSerializer(tutor_qs, many=True).data,
        })
