from django.http import JsonResponse
from django.views.decorators.http import require_GET, require_POST
from django.views.decorators.csrf import csrf_exempt
from django.contrib.auth import get_user_model

from .models import SupportThread


def _ensure_session(request) -> str:
    if not request.session.session_key:
        request.session.save()
    return request.session.session_key or ""


def _pick_staff():
    User = get_user_model()
    return User.objects.filter(is_staff=True, is_active=True).order_by("id").first()


@require_POST
@csrf_exempt
def support_thread(request):
    """Return (and create if needed) the current user's or guest's support thread.
    Also auto-assigns a staff user when possible.
    """
    user = request.user if getattr(request, "user", None) and request.user.is_authenticated else None

    if user:
        thread, _ = SupportThread.objects.get_or_create(user=user, status="open")
    else:
        sk = _ensure_session(request)
        thread, _ = SupportThread.objects.get_or_create(session_key=sk, status="open")

    # Auto-assign to a staff user (first available) if not assigned
    if not thread.assigned_staff_id:
        try:
            staff = _pick_staff()
            if staff:
                thread.assigned_staff = staff
                thread.save(update_fields=["assigned_staff", "updated_at"])
        except Exception:
            pass

    return JsonResponse({"thread_id": thread.id})


@require_GET
def support_history(request, thread_id: int):
    """Return last messages for a thread (owner/guest/staff)."""
    user = request.user if getattr(request, "user", None) and request.user.is_authenticated else None
    sk = request.session.session_key or ""

    try:
        thread = SupportThread.objects.get(id=thread_id)
    except SupportThread.DoesNotExist:
        return JsonResponse({"detail": "Not found"}, status=404)

    allowed = False
    if user and user.is_staff:
        allowed = True
    elif user and thread.user_id == user.id:
        allowed = True
    elif (not user) and sk and thread.session_key == sk:
        allowed = True

    if not allowed:
        return JsonResponse({"detail": "Forbidden"}, status=403)

    msgs = thread.messages.all().order_by("-created_at")[:30]
    data = []
    for m in reversed(list(msgs)):
        sender_name = "Guest"
        if m.sender_label == "staff":
            sender_name = (m.sender.get_full_name() or m.sender.username or "Support") if m.sender else "Support"
        elif m.sender_label == "user":
            sender_name = (m.sender.get_full_name() or m.sender.username or "User") if m.sender else "User"

        data.append(
            {
                "id": m.id,
                "sender_label": m.sender_label,
                "sender_name": sender_name,
                "text": m.text,
                "created_at": m.created_at.isoformat(),
            }
        )

    return JsonResponse({"thread_id": thread.id, "messages": data})


@require_POST
@csrf_exempt
def support_mark_read(request, thread_id: int):
    user = request.user if getattr(request, "user", None) and request.user.is_authenticated else None
    sk = request.session.session_key or ""

    try:
        thread = SupportThread.objects.get(id=thread_id)
    except SupportThread.DoesNotExist:
        return JsonResponse({"detail": "Not found"}, status=404)

    allowed = False
    if user and user.is_staff:
        allowed = True
    elif user and thread.user_id == user.id:
        allowed = True
    elif (not user) and sk and thread.session_key == sk:
        allowed = True

    if not allowed:
        return JsonResponse({"detail": "Forbidden"}, status=403)

    # mark staff messages as read for owner; mark user/guest as read for staff
    if user and user.is_staff:
        thread.messages.filter(sender_label__in=["user", "guest"]).update(is_read=True)
    else:
        thread.messages.filter(sender_label="staff").update(is_read=True)

    return JsonResponse({"ok": True})
