from django.shortcuts import render, redirect
from django.contrib.auth.decorators import login_required
from django.shortcuts import redirect, render
from bookings.models import Booking
from accounts.models import User

@login_required
def router(request):
    if request.user.role == User.ROLE_TUTOR:
        return redirect("portal:tutor_dashboard")
    return redirect("portal:student_dashboard")

@login_required
def student_dashboard(request):
    bookings = Booking.objects.filter(student=request.user).order_by("-created_at")[:50]
    return render(request, "portal/student/index.html", {"bookings": bookings})

@login_required
def tutor_dashboard(request):
    bookings = Booking.objects.filter(tutor=request.user).order_by("-created_at")[:50]
    return render(request, "portal/tutor/index.html", {"bookings": bookings})

@login_required
def new_booking(request):
    # UI-only steps handled in templates with JS + API
    return render(request, "portal/student/new-booking.html")

@login_required
def booking_step2(request):
    return render(request, "portal/student/booking-step2.html")

@login_required
def booking_step3(request):
    return render(request, "portal/student/booking-step3.html")

@login_required
def booking_confirm(request):
    return render(request, "portal/student/booking-confirm.html")

@login_required
def bookings_list(request):
    bookings = Booking.objects.filter(student=request.user).order_by("-created_at")
    return render(request, "portal/student/bookings.html", {"bookings": bookings})


from django.shortcuts import get_object_or_404
from chat.models import Conversation

@login_required
def student_chats(request):
    convs = Conversation.objects.filter(student=request.user).order_by("-created_at")
    return render(request, "portal/student/chats.html", {"conversations": convs})

@login_required
def tutor_chats(request):
    convs = Conversation.objects.filter(tutor=request.user).order_by("-created_at")
    return render(request, "portal/tutor/chats.html", {"conversations": convs})

@login_required
def student_chat_room(request, booking_id: int):
    booking = get_object_or_404(Booking, id=booking_id, student=request.user)
    if not booking.tutor_id:
        return redirect("portal:student_dashboard")
    conv, _ = Conversation.objects.get_or_create(booking=booking, defaults={"student": booking.student, "tutor": booking.tutor})
    return render(request, "portal/student/chat-room.html", {"conversation": conv, "booking": booking})

@login_required
def tutor_chat_room(request, booking_id: int):
    booking = get_object_or_404(Booking, id=booking_id, tutor=request.user)
    conv, _ = Conversation.objects.get_or_create(booking=booking, defaults={"student": booking.student, "tutor": booking.tutor})
    return render(request, "portal/tutor/chat-room.html", {"conversation": conv, "booking": booking})


from django.contrib.admin.views.decorators import staff_member_required
from django.db.models import Q
from chat.models import SupportThread, SupportMessage

@staff_member_required
def staff_support_inbox(request):
    q = (request.GET.get("q") or "").strip()
    status = (request.GET.get("status") or "").strip()

    qs = SupportThread.objects.all().order_by("-updated_at")
    if status:
        qs = qs.filter(status=status)

    if q:
        qs = qs.filter(
            Q(user__username__icontains=q) |
            Q(session_key__icontains=q) |
            Q(messages__text__icontains=q)
        ).distinct()

    threads = []
    for t in qs[:200]:
        last = t.messages.order_by("-created_at").first()
        t.last_preview = (last.text[:90] + "…") if last and len(last.text) > 90 else (last.text if last else "")
        threads.append(t)

    return render(request, "portal/staff/support-inbox.html", {"threads": threads, "q": q, "status": status})

@staff_member_required
def staff_support_thread(request, thread_id: int):
    thread = SupportThread.objects.get(id=thread_id)
    messages = thread.messages.all().order_by("created_at")[:200]
    return render(request, "portal/staff/support-thread.html", {"thread": thread, "messages": messages})
