// Register Service Worker
if ("serviceWorker" in navigator) {
  navigator.serviceWorker.register("/static/js/sw.js").catch(()=>{});
}

// Simple forms (frontend-only placeholders)
function wireLeadForm(formId, msgId, endpoint) {
  const form = document.getElementById(formId);
  const msg = document.getElementById(msgId);
  if (!form) return;

  form.addEventListener("submit", async (e) => {
    e.preventDefault();
    if (msg) msg.textContent = "Submitting...";

    const fd = new FormData(form);
    // useful context
    if (!fd.get("page_url")) fd.set("page_url", window.location.href);

    try {
      if (endpoint) {
        const resp = await fetch(endpoint, { method: "POST", body: fd });
        const data = await resp.json().catch(() => ({}));
        if (!resp.ok || !data.ok) {
          const err = (data && data.error) ? data.error : "Submission failed. Check email settings.";
          if (msg) msg.textContent = "❌ " + err;
          return;
        }
      } else {
        // fallback: UI-only success
        await new Promise(r => setTimeout(r, 400));
      }

      form.reset();
      if (msg) msg.textContent = "✅ Submitted! We will contact you shortly.";
    } catch (err) {
      if (msg) msg.textContent = "❌ Network error. Please try again.";
    }
  });
}

wireLeadForm("leadForm", "leadMsg", "/api/lead/");
wireLeadForm("consultForm", "consultMsg", "/api/consultation/");




// Make landing/portal cards fully clickable (native feel) without changing HTML layout
document.addEventListener("DOMContentLoaded", () => {
  const cardSelectors = [
    ".card-soft",
    ".card",
    ".native-card",
    ".pricing-card",
    ".exam-card",
    ".service-card",
    ".list-card"
  ];

  const cards = document.querySelectorAll(cardSelectors.join(","));
  cards.forEach((card) => {
    // ignore if already wired
    if (card.dataset.cardWired === "1") return;

    // find first meaningful link inside the card
    const link = card.querySelector('a[href]:not([href=""]):not([href="#"])');
    if (!link) return;

    const href = link.getAttribute("href");
    if (!href || href.startsWith("javascript:")) return;

    // make the whole card clickable
    card.classList.add("is-clickable");
    card.setAttribute("role", "link");
    card.setAttribute("tabindex", "0");
    card.dataset.cardHref = href;
    card.dataset.cardWired = "1";

    // avoid breaking normal link clicks
    card.addEventListener("click", (e) => {
      const isAnchor = e.target && (e.target.closest && e.target.closest("a"));
      const isButton = e.target && (e.target.closest && e.target.closest("button"));
      const isForm = e.target && (e.target.closest && e.target.closest("form, input, select, textarea, label"));
      if (isAnchor || isButton || isForm) return;
      window.location.href = href;
    });

    card.addEventListener("keydown", (e) => {
      if (e.key === "Enter" || e.key === " ") {
        e.preventDefault();
        window.location.href = href;
      }
    });
  });
});


// Floating Support / Chat sheet
document.addEventListener("DOMContentLoaded", () => {
  const fab = document.getElementById("supportFab");
  const sheet = document.getElementById("supportSheet");
  if (!fab || !sheet) return;

  const closeBtn = sheet.querySelector("[data-support-close]");
  const setOpen = (open) => {
    sheet.classList.toggle("open", open);
    sheet.setAttribute("aria-hidden", open ? "false" : "true");
  };

  fab.addEventListener("click", (e) => {
    e.preventDefault();
    setOpen(!sheet.classList.contains("open"));
  });

  if (closeBtn) {
    closeBtn.addEventListener("click", (e) => {
      e.preventDefault();
      setOpen(false);
    });
  }

  // Close when clicking outside
  document.addEventListener("click", (e) => {
    if (!sheet.classList.contains("open")) return;
    const inside = sheet.contains(e.target) || fab.contains(e.target);
    if (!inside) setOpen(false);
  });

  // Close on Escape
  document.addEventListener("keydown", (e) => {
    if (e.key === "Escape") setOpen(false);
  });
});



/* ==========================================================
   ET_CHAT_BOT_V1 - guided support chat + real support inbox (Channels)
   - Button label: ET-chat
   - Sends user messages into /chat/support/thread/ (POST) + /ws/support/<id>/
   - Staff replies show badge + Android sound/vibrate when available
========================================================== */
(function () {
  const $ = (sel) => document.querySelector(sel);
  const $$ = (sel) => Array.from(document.querySelectorAll(sel));

  const openBtn = $("#openEtChat");
  const widget = $("#etChatWidget");
  const closeBtn = $("#closeEtChat");
  const body = $("#etChatBody");
  const quick = $("#etChatQuick");
  const form = $("#etChatForm");
  const input = $("#etChatInput");
  const badgeEl = $("#etChatBadge");
  const statusEl = $("#etChatStatus");

  if (!openBtn || !widget || !closeBtn || !body || !quick || !form || !input) return;

  const STORAGE_KEY = "et_chat_history_v1";
  const BADGE_KEY = "et_chat_unread_v1";
  const THREAD_KEY = "et_chat_thread_v1";

  // ---------------- Badge ----------------
  function setBadge(n) {
    if (!badgeEl) return;
    const v = Math.max(0, parseInt(n || 0, 10) || 0);
    badgeEl.textContent = String(v);
    badgeEl.style.display = v > 0 ? "grid" : "none";
    try { localStorage.setItem(BADGE_KEY, String(v)); } catch (e) {}
  }
  function incBadge() {
    try {
      const n = parseInt(localStorage.getItem(BADGE_KEY) || "0", 10) || 0;
      setBadge(n + 1);
    } catch (e) {
      setBadge(1);
    }
  }
  function loadBadge() {
    try { setBadge(parseInt(localStorage.getItem(BADGE_KEY) || "0", 10) || 0); } catch (e) {}
  }
  loadBadge();

  // ---------------- Bot intents (few, strong) ----------------
  const BOT = {
    greeting: "Hi 👋 I’m ET-chat. What do you need help with?",
    intents: [
      {
        key: "pricing",
        keywords: ["price", "pricing", "plan", "fee", "cost", "how much"],
        answer:
          "Pricing ✅\nTell me what you need (Subject tutoring / Coding / Exam prep) + your level.\nI’ll guide you to the right plan.",
        chips: ["Book a tutor", "Exams", "Contact"]
      },
      {
        key: "booking",
        keywords: ["book", "booking", "tutor", "session", "schedule", "class", "lesson"],
        answer:
          "Booking ✅\n1) Open Booking\n2) Pick subject/exam\n3) Choose time\n4) Submit — a tutor can accept/decline.\n\nReply with: subject + exam + your timezone.",
        chips: ["Pricing", "Exams", "Student Portal"]
      },
      {
        key: "exams",
        keywords: ["exam", "waec", "jamb", "neco", "igcse", "sat", "ielts", "toefl", "gmat", "gre"],
        answer:
          "Exam prep ✅\nTell me the exam + weak topics. I’ll suggest a study plan.",
        chips: ["Book a tutor", "Pricing", "Contact"]
      },
      {
        key: "contact",
        keywords: ["contact", "email", "phone", "whatsapp", "support", "help"],
        answer:
          "Contact ✅\n• WhatsApp: Tap WhatsApp\n• Email: Tap Email",
        chips: ["WhatsApp", "Email"]
      },
      {
        key: "whatsapp",
        keywords: ["whatsapp", "wa"],
        answer: "Opening WhatsApp…",
        action: { type: "click", selector: "#supportWhatsapp" },
        chips: ["Email", "Pricing"]
      },
      {
        key: "email",
        keywords: ["email", "mail"],
        answer: "Opening Email…",
        action: { type: "click", selector: "#supportEmail" },
        chips: ["WhatsApp", "Pricing"]
      },
      {
        key: "student_portal",
        keywords: ["student portal", "student dashboard", "student", "portal"],
        answer: "Opening Student Portal…",
        action: { type: "navigate", path: "/portal/student/" },
        chips: ["Book a tutor", "Contact"]
      },
      {
        key: "tutor_portal",
        keywords: ["tutor portal", "tutor dashboard", "tutor", "teacher"],
        answer: "Opening Tutor Portal…",
        action: { type: "navigate", path: "/portal/tutor/" },
        chips: ["Contact", "Pricing"]
      }
    ],
    fallback:
      "I can help with: pricing, booking, exams, portals, and contact.\nTap a quick option below."
  };

  // ---------------- UI helpers ----------------
  function escapeHtml(s) {
    return (s || "").replace(/[&<>"']/g, (c) => ({ "&":"&amp;","<":"&lt;",">":"&gt;",'"':"&quot;","'":"&#39;" }[c]));
  }

  function saveHistory() {
    try {
      const msgs = [];
      $$(".et-chat-body .et-msg").forEach((node) => {
        const role = node.classList.contains("user") ? "user" : "bot";
        const bub = node.querySelector(".et-bub");
        const text = (bub ? bub.dataset.raw || bub.textContent : "").trim();
        if (text) msgs.push({ role, text });
      });
      localStorage.setItem(STORAGE_KEY, JSON.stringify(msgs.slice(-80)));
    } catch (e) {}
  }

  function loadHistory() {
    try {
      const data = JSON.parse(localStorage.getItem(STORAGE_KEY) || "[]");
      if (!Array.isArray(data) || !data.length) return false;
      body.innerHTML = "";
      data.slice(-60).forEach((m) => pushMsg(m.role, m.text, false));
      body.scrollTop = body.scrollHeight;
      return true;
    } catch (e) {
      return false;
    }
  }

  function pushMsg(role, text, doSave = true) {
    const wrap = document.createElement("div");
    wrap.className = "et-msg " + (role === "user" ? "user" : "bot");

    const bub = document.createElement("div");
    bub.className = "et-bub";
    bub.dataset.raw = text || "";
    bub.innerHTML = escapeHtml(text || "").replace(/\n/g, "<br/>");

    wrap.appendChild(bub);
    body.appendChild(wrap);
    body.scrollTop = body.scrollHeight;
    if (doSave) saveHistory();
  }

  function pushUserWithTicks(text, clientId) {
    const wrap = document.createElement("div");
    wrap.className = "et-msg user";
    wrap.dataset.clientId = clientId || "";

    const bub = document.createElement("div");
    bub.className = "et-bub";
    bub.dataset.raw = text || "";
    bub.innerHTML = escapeHtml(text || "").replace(/\n/g, "<br/>");

    const ticks = document.createElement("div");
    ticks.className = "et-ticks";
    ticks.textContent = "…"; // pending
    bub.appendChild(ticks);

    wrap.appendChild(bub);
    body.appendChild(wrap);
    body.scrollTop = body.scrollHeight;
    saveHistory();
  }

  function markDelivered(clientId) {
    if (!clientId) return;
    const node = body.querySelector(`.et-msg.user[data-client-id="${clientId}"] .et-ticks`);
    if (node) node.textContent = "✓";
  }

  function showTyping(show) {
    const id = "etTypingRow";
    let row = document.getElementById(id);
    if (!show) {
      if (row) row.remove();
      return;
    }
    if (row) return;
    row = document.createElement("div");
    row.id = id;
    row.className = "et-msg bot";
    const bub = document.createElement("div");
    bub.className = "et-bub";
    bub.textContent = "Typing…";
    row.appendChild(bub);
    body.appendChild(row);
    body.scrollTop = body.scrollHeight;
  }

  function setChips(labels) {
    quick.innerHTML = "";
    (labels || []).slice(0, 6).forEach((label) => {
      const b = document.createElement("button");
      b.type = "button";
      b.className = "et-chip";
      b.textContent = label;
      b.addEventListener("click", () => {
        handleUser(label);
      });
      quick.appendChild(b);
    });
  }

  function runAction(action) {
    if (!action) return;
    if (action.type === "navigate" && action.path) {
      window.location.href = action.path;
    }
    if (action.type === "click" && action.selector) {
      const el = document.querySelector(action.selector);
      if (el) el.click();
    }
  }

  function bestIntent(text) {
    const t = (text || "").toLowerCase();
    for (const intent of BOT.intents) {
      if (intent.keywords.some((k) => t.includes(k))) return intent;
    }
    return null;
  }

  // ---------------- Support inbox via WS ----------------
  let supportThreadId = null;
  let supportWS = null;
  let supportWSReady = false;
  let typingTimer = null;

  function setStatus(s) {
    if (!statusEl) return;
    statusEl.textContent = s;
  }

  function getWSUrl(path) {
    const proto = window.location.protocol === "https:" ? "wss://" : "ws://";
    return proto + window.location.host + path;
  }

  async function ensureSupportThread() {
    try {
      const cached = localStorage.getItem(THREAD_KEY);
      if (!supportThreadId && cached) supportThreadId = parseInt(cached, 10) || null;
    } catch (e) {}
    if (supportThreadId) return supportThreadId;

    try {
      const resp = await fetch("/chat/support/thread/", {
        method: "POST",
        headers: { "X-Requested-With": "XMLHttpRequest" }
      });
      if (!resp.ok) throw new Error("thread");
      const data = await resp.json();
      supportThreadId = data.thread_id;
      try { localStorage.setItem(THREAD_KEY, String(supportThreadId)); } catch (e) {}
      return supportThreadId;
    } catch (e) {
      supportThreadId = null;
      return null;
    }
  }

  async function loadSupportHistory(threadId) {
    try {
      const r = await fetch(`/chat/support/${threadId}/history/`, { headers: { "X-Requested-With": "XMLHttpRequest" } });
      if (!r.ok) return;
      const data = await r.json();
      if (Array.isArray(data.messages) && data.messages.length) {
        data.messages.forEach((m) => {
          if (m.type === "typing") return;
          const role = m.sender_label === "staff" ? "bot" : "user";
          pushMsg(role, m.text, false);
        });
        body.scrollTop = body.scrollHeight;
        saveHistory();
      }
      // clear unread badge once chat is opened
      try { await fetch(`/chat/support/${threadId}/mark_read/`, { method: "POST", headers: { "X-Requested-With": "XMLHttpRequest" } }); } catch (e) {}
      setBadge(0);
    } catch (e) {}
  }

  async function connectSupportWS(threadId) {
    if (!threadId) return;
    if (supportWS && supportWSReady) return;

    try {
      supportWS = new WebSocket(getWSUrl(`/ws/support/${threadId}/`));
      supportWSReady = false;

      supportWS.onopen = () => {
        supportWSReady = true;
        setStatus("Online • connected");
      };
      supportWS.onclose = () => {
        supportWSReady = false;
        setStatus("Reconnecting…");
      };
      supportWS.onerror = () => {
        supportWSReady = false;
        setStatus("Connection issue…");
      };

      supportWS.onmessage = (ev) => {
        try {
          const m = JSON.parse(ev.data);

          // typing
          if (m.type === "typing") {
            if (m.sender_label === "staff") showTyping(!!m.is_typing);
            return;
          }

          // delivery ack
          if (m.client_id) markDelivered(m.client_id);

          // only render staff messages from socket (avoid duplicating our own)
          if (m.sender_label === "staff") {
            showTyping(false);
            pushMsg("bot", m.text);
            setChips(["Pricing", "Book a tutor", "Exams", "Contact"]);

            // badge if widget closed
            if (!widget.classList.contains("is-open")) {
              incBadge();
              // Android sound/vibrate bridge (if present)
              try { window.AndroidBridge && window.AndroidBridge.notifySupportReply && window.AndroidBridge.notifySupportReply(); } catch (e) {}
            }
          }
        } catch (e) {}
      };
    } catch (e) {}
  }

  function sendTyping(isTyping) {
    if (!supportWS || !supportWSReady) return;
    try { supportWS.send(JSON.stringify({ type: "typing", is_typing: !!isTyping })); } catch (e) {}
  }

  async function sendToSupportInbox(text, clientId) {
    const threadId = await ensureSupportThread();
    if (!threadId) return; // backend not available
    await connectSupportWS(threadId);
    if (supportWS && supportWSReady) {
      try { supportWS.send(JSON.stringify({ type: "message", text, client_id: clientId || "" })); } catch (e) {}
    }
  }

  // ---------------- Main message handler ----------------
  function handleUser(raw) {
    const text = (raw || "").trim();
    if (!text) return;

    const clientId = "c_" + Date.now() + "_" + Math.random().toString(16).slice(2);
    pushUserWithTicks(text, clientId);
    showTyping(true);

    // send into real support inbox (staff can reply)
    sendToSupportInbox(text, clientId);

    // lightweight bot response (instant) – still keep staff inbox for real replies
    const intent = bestIntent(text);
    const reply = intent ? intent.answer : BOT.fallback;

    setTimeout(() => {
      showTyping(false);
      pushMsg("bot", reply);
      setChips(intent?.chips || ["Pricing", "Book a tutor", "Exams", "Contact"]);
      runAction(intent?.action);
    }, 420);
  }

  // ---------------- Open/close ----------------
  async function openChat() {
    widget.classList.add("is-open");
    widget.setAttribute("aria-hidden", "false");

    // prime thread + history
    const tid = await ensureSupportThread();
    if (tid) {
      await loadSupportHistory(tid);
      await connectSupportWS(tid);
    }

    // greet once if empty
    const hasHistory = loadHistory();
    if (!hasHistory && body.children.length === 0) {
      pushMsg("bot", BOT.greeting);
      setChips(["Pricing", "Book a tutor", "Exams", "Contact"]);
    }

    // mark read
    if (tid) {
      try { await fetch(`/chat/support/${tid}/mark_read/`, { method: "POST", headers: { "X-Requested-With": "XMLHttpRequest" } }); } catch (e) {}
      setBadge(0);
    }

    setTimeout(() => input.focus(), 80);
  }

  function closeChat() {
    widget.classList.remove("is-open");
    widget.setAttribute("aria-hidden", "true");
    showTyping(false);
  }

  openBtn.addEventListener("click", openChat);
  closeBtn.addEventListener("click", closeChat);

  input.addEventListener("input", () => {
    if (typingTimer) clearTimeout(typingTimer);
    sendTyping(true);
    typingTimer = setTimeout(() => sendTyping(false), 900);
  });

  form.addEventListener("submit", (e) => {
    e.preventDefault();
    const v = (input.value || "").trim();
    if (!v) return;
    input.value = "";
    handleUser(v);
  });

  // CTRL/CMD+Enter clears local chat (dev tool)
  input.addEventListener("keydown", (e) => {
    if (e.key === "Enter" && (e.ctrlKey || e.metaKey)) {
      e.preventDefault();
      try { localStorage.removeItem(STORAGE_KEY); } catch (e2) {}
      body.innerHTML = "";
      pushMsg("bot", BOT.greeting);
      setChips(["Pricing", "Book a tutor", "Exams", "Contact"]);
    }
  });
})();

/* ==========================================================
   ET_CHAT_DELEGATION_TOGGLE_V1
   Makes ET button clickable even when header is loaded dynamically
========================================================== */
(function(){
  const toggleSheet = () => {
    const sheet = document.getElementById("supportSheet");
    if (!sheet) return;
    sheet.classList.toggle("open");
    sheet.setAttribute("aria-hidden", sheet.classList.contains("open") ? "false" : "true");
  };

  document.addEventListener("click", (e) => {
    const fab = e.target.closest && e.target.closest("#supportFab");
    if (!fab) return;
    e.preventDefault();
    toggleSheet();
  });
})();
/* ET_CHAT_TYPING_INPUT_V1 */

/* ET_CHAT_ENHANCE_V2 */
